<?php
session_start();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (is_csrf_valid()) {
        include_once includes('config.php');
        include_once includes('mailer/send.php');

        // Process the password reset request
        if (isset($_POST['email'])) {
            $email = trim($_POST['email']);
            if (isRecentRequest($email, $conn)) {
                // Return an error response indicating rate limiting
                $response = array('status' => 'error', 'message' => $lang['user-password-reset-limit-error']);
                echo json_encode($response);
                exit;
            }

            $sql = "SELECT * FROM `users` WHERE email = ?";
            $stmt = mysqli_prepare($conn, $sql);
            mysqli_stmt_bind_param($stmt, "s", $email);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);

            if (mysqli_num_rows($result) == 1) {
                $row = mysqli_fetch_assoc($result);
                // Check if user registered with Google
                if ($row['google_uid']) {
                    // User registered with Google, display message
                    $response = array('status' => 'error', 'message' => $lang['user-password-reset-google-error']);
                    echo json_encode($response);
                    exit;
                }
                // Email exists and not registered with Google, continue with password reset
                // Generate and store a reset token (e.g., in a separate reset_tokens table)
                $resetToken = generateResetToken();
                $expiryTime = date("Y-m-d H:i:s", strtotime("+7 days"));

                // Send the reset link to the user's email address
                // Retrieve the user's name from the database
                $userName = $row['name']; // Assuming 'name' is the column name for the user's name
                $emailSubject = $lang['email-reset-password-subject'];
                $emailContent = getResetEmailContent($resetToken, $userName, $lang);
                // Send the reset link to the user's email address with their name
                sendEmail($email, $emailSubject, $emailContent, '');
                $success = storeResetToken($email, $resetToken, $expiryTime, $conn);

                if ($success) {
                    // Return success response to the JavaScript code
                    $response = array('status' => 'success', 'message' => $lang['user-password-reset-link-sent']);
                    echo json_encode($response);
                } else {
                    // Email sending failed
                    $response = array('status' => 'error', 'message' => $lang['user-password-reset-link-failed']);
                    echo json_encode($response);
                }
            } else {
                // User not found
                // Email does not exist, return error response to the JavaScript code
                $response = array('status' => 'error', 'message' => $lang['user-password-reset-email-not-found']);
                echo json_encode($response);
            }
        } else {
            // Invalid request, return error response to the JavaScript code
            $response = array('status' => 'error', 'message' => $lang['user-password-reset-invalid-request']);
            echo json_encode($response);
        }
    } else {
        // CSRF token validation failed
        $response = array('status' => 'error', 'message' => 'CSRF token validation failed.');
        echo json_encode($response);
    }
}
// Function to generate the email content for the password reset email
function getResetEmailContent($resetToken, $userName, $lang) {
    $htmlContent = file_get_contents(includes('emails/system_email.php'));

    $dynamicContent = generateResetLink($resetToken);

    // Replace placeholders with dynamic content
    $replacements = [
        '{{cta-link}}' => $dynamicContent,
        '{{user-name}}' => $userName,
        '{{headline}}' => $lang['email-reset-password-headline'],
        '{{email-p1}}' => $lang['email-reset-instructions'],
        '{{cta}}' => $lang['email-reset-password-cta'],
        '{{email-p2}}' => $lang['email-reset-instructions-p2'],
        '{{email-p3}}' => '',

        // Language-specific replacements
        '{{hello}}' => $lang['email-hello'] ?? '',
        '{{email-copyright}}' => $lang['email-copyright'] ?? '',
        '{{best-regards}}' => $lang['best-regards'] ?? 'Best regards',
        '{{company-team}}' => $lang['company-team'] ?? 'The Datanito Team',
        '{{home}}' => $lang['email-home'] ?? 'Home',
        '{{about}}' => $lang['email-about'] ?? 'About',
        '{{products}}' => $lang['email-ai'] ?? 'Datanito AI',
        '{{contact}}' => $lang['email-contact'] ?? 'Contact',
        '{{privacy}}' => $lang['email-privacy'] ?? 'Privacy',
        '{{company-name}}' => $lang['company-name'] ?? 'Datanito Ltd.',
        '{{company-tagline}}' => $lang['company-tagline'] ?? 'Leading AI-Powered Software and Payment Solutions'
    ];

    foreach ($replacements as $placeholder => $value) {
        $htmlContent = str_replace($placeholder, $value, $htmlContent);
    }

    return $htmlContent;
}

function isRecentRequest($email, $conn) {
    // Get the current timestamp
    $currentTime = time();

    // Calculate the cutoff time for recent requests (1 minute, 1 hour, or 1 day ago)
    $oneMinuteAgo = $currentTime - 60; // 60 seconds in 1 minute
    $oneHourAgo = $currentTime - 3600; // 3600 seconds in 1 hour
    $oneDayAgo = $currentTime - 86400; // 86400 seconds in 1 day

    // Query to retrieve timestamps of recent requests
    $query = "SELECT created_at FROM `password_reset_tokens` WHERE email = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "s", $email);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);

    // Check each timestamp to determine if it falls within the cutoff time
    $recentRequests = 0;
    while ($row = mysqli_fetch_assoc($result)) {
        $requestTime = strtotime($row['created_at']);
        if ($requestTime >= $oneMinuteAgo) {
            $recentRequests++;
        }
    }

    // Determine if the number of recent requests exceeds the limit for the time window
    if ($recentRequests >= 5) {
        return true; // If 5 or more requests have been made in the past day
    } elseif ($recentRequests >= 3) {
        return true; // If 3 or more requests have been made in the past hour
    } else {
        return false; // If less than 3 requests have been made in the past hour
    }
}


// Function to generate a reset token
function generateResetToken() {
    // Generate and return a random reset token
    $length = 20;
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $token = '';
    for ($i = 0; $i < $length; $i++) {
        $token .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $token;
}

// Function to generate the password reset link
function generateResetLink($resetToken) {
    // Generate and return the password reset link based on the reset token
    $resetLink = url('/account/update-password/'. urlencode($resetToken));
    return $resetLink;
}

// Function to store the reset token in the database
function storeResetToken($email, $resetToken, $expiryTime, $conn) {
    // Define the query to insert the reset token into the reset_tokens table
    $sqlReset = "INSERT INTO `password_reset_tokens` (email, token, expiry_time) VALUES (?, ?, ?)";
    $stmtReset = mysqli_prepare($conn, $sqlReset);
    mysqli_stmt_bind_param($stmtReset, "sss", $email, $resetToken, $expiryTime);

    // Execute the INSERT query
    $success = mysqli_stmt_execute($stmtReset);

    // Close the statement and the database connection
    mysqli_stmt_close($stmtReset);
    mysqli_close($conn);

    return $success;
}
