<?php

session_start();


// Check if the request method is POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check CSRF token validity
    if (is_csrf_valid()) {
        // Include necessary files
        include_once includes('config.php');
        include_once includes('mailer/send.php');

        // Process the registration request
        if (isset($_POST['email']) && isset($_POST['name']) && isset($_POST['username']) && isset($_POST['password'])) {
            // Extract form data
            $email = trim($_POST['email']);
            $name = trim($_POST['name']);
            $username = trim($_POST['username']);
            $password = trim($_POST['password']);
            $currency = $_SESSION["currency"];
            $region = isset($_SESSION["country"]) ? $_SESSION["country"] : 'us'; // Get preferred country from session

            // Generate a random and untaken user_id
            $user_id = generateRandomUserId(1000000000, 9999999999, $conn);

            // Encrypt the password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            // Validate form inputs
            if (!$name || strlen($password) < 8 || strlen($username) < 4 || strlen($username) > 32 || !preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
                $response = array('status' => 'error', 'message' => 'Invalid form data');
                echo json_encode($response);
                die();
            }

            // Check if the email or username is already taken
            $sqlCheck = "SELECT * FROM `users` WHERE email = ? OR username = ?";
            $stmtCheck = mysqli_prepare($conn, $sqlCheck);
            mysqli_stmt_bind_param($stmtCheck, "ss", $email, $username);
            mysqli_stmt_execute($stmtCheck);
            $resultCheck = mysqli_stmt_get_result($stmtCheck);

            $foundEmail = false;
            $foundUsername = false;

            while ($row = mysqli_fetch_assoc($resultCheck)) {
                if ($row['email'] === $email) {
                    $foundEmail = true;
                }
                if ($row['username'] === $username) {
                    $foundUsername = true;
                }
            }

            // Handle duplicate email/username
            if ($foundEmail && $foundUsername) {
                $response = array('status' => 'error', 'message' => 'Email and username already taken');
                echo json_encode($response);
                die();
            } elseif ($foundEmail) {
                $response = array('status' => 'error', 'message' => 'Email already taken');
                echo json_encode($response);
                die();
            } elseif ($foundUsername) {
                // Suggest a related username
                $suggestedUsername = suggestRelatedUsername($username);
                $response = array('status' => 'error', 'message' => 'Username already taken', 'suggested_username' => $suggestedUsername);
                echo json_encode($response);
                die();
            }

            // Insert the user into the database
            $sql = "INSERT INTO users (user_id, email, name, username, password, currency, preferred_region) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt = mysqli_prepare($conn, $sql);
            mysqli_stmt_bind_param($stmt, "issssss",  $user_id, $email, $name, $username, $hashed_password, $currency, $region);
            mysqli_stmt_execute($stmt);

            if (mysqli_stmt_affected_rows($stmt) > 0) {
                // Generate a verification token
                $verificationToken = generateVerificationToken();

                // Store the token in the email_verification table
                $insertTokenSql = "INSERT INTO email_verification (user_id, token) VALUES (?, ?)";
                $insertTokenStmt = mysqli_prepare($conn, $insertTokenSql);
                mysqli_stmt_bind_param($insertTokenStmt, "is", $user_id, $verificationToken);
                mysqli_stmt_execute($insertTokenStmt);

                // Create a verification link based on the token
                $verificationLink = url('/account/verify/'. urlencode($verificationToken));

                // Prepare verification email content
                $emailSubject = $lang['email-verification-subject'];
                $emailContent = getVerificationEmailContent($verificationLink, $name, $lang);

                // Send the verification email
                sendEmail($email, $emailSubject, $emailContent, '');

                // Set session variables
                $_SESSION['user']['id'] = $user_id;
                $_SESSION['userLoggedIn'] = true;

                // Generate and save remember me token
                $token = generateToken();
                $expires_at = date('Y-m-d H:i:s', strtotime('+30 days')); // Token expiration set to 30 days from now
                saveTokenInDatabase($user_id, $token, $expires_at, $conn);
                setcookie('remember_token', $token, time() + (60 * 60 * 24 * 30), '/'); // Cookie expires in 30 days

                // Return success response to the JavaScript code
                $response = array('status' => 'success', 'message' => 'User registered successfully');
                echo json_encode($response);
            } else {
                // Handle database query execution failure
                $error = mysqli_stmt_error($stmt);
                $response = array('status' => 'error', 'message' => 'Registration failed: ' . $error);
                echo json_encode($response);
            }
        } elseif (isset($_POST['email']) && $_POST['action'] == 'validate_email') {
            // Process the email submission request
            $email = $_POST['email'];
            // You can perform additional validation here before storing the email
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $response = array('status' => 'error', 'message' => 'Invalid email format');
            } else {
                $response = array('status' => 'success');
            }
            echo json_encode($response);
        }

        // Close the database connection
        mysqli_close($conn);
    } else {
        // handle CSRF token validation failure
        echo 'CSRF token validation failed';
    }
}
function generateVerificationToken() {
    // Generate and return a random verification token
    $length = 32;
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $token = '';
    for ($i = 0; $i < $length; $i++) {
        $token .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $token;
}

// Function to generate a random and untaken user_id
function generateRandomUserId($min, $max, $conn) {
    do {
        // Generate a random user ID within the specified range
        $user_id = strval(mt_rand($min, $max));

        // Pad the user ID to ensure it is 10 characters long
        $user_id = str_pad($user_id, 10, '0', STR_PAD_LEFT);

        // Check if the generated user ID is unique
        $sqlCheck = "SELECT COUNT(*) FROM `users` WHERE user_id = ?";
        $stmtCheck = mysqli_prepare($conn, $sqlCheck);

        // Check if the prepare statement is successful
        if ($stmtCheck) {
            mysqli_stmt_bind_param($stmtCheck, "s", $user_id);
            mysqli_stmt_execute($stmtCheck);
            mysqli_stmt_bind_result($stmtCheck, $count);
            mysqli_stmt_fetch($stmtCheck);
            mysqli_stmt_close($stmtCheck);
        } else {
            // Handle prepare statement failure
            die('mysqli_prepare failed');
        }
    } while ($count > 0);

    return $user_id;
}


function getVerificationEmailContent($verificationLink, $userName, $lang) {
    $htmlContent = file_get_contents(includes('emails/system_email.php'));

    // Replace placeholders with dynamic content
    $replacements = [
        '{{cta-link}}' => $verificationLink,
        '{{user-name}}' => $userName,
        '{{headline}}' => $lang['email-verification-headline'],
        '{{email-p1}}' => $lang['email-verification-instructions'],
        '{{cta}}' => $lang['email-verification-cta'],
        '{{email-p2}}' => $lang['email-verification-instructions-p2'],
        '{{email-p3}}' => '',

        // Language-specific replacements
        '{{hello}}' => $lang['email-hello'] ?? '',
        '{{email-copyright}}' => $lang['email-copyright'] ?? '',
        '{{best-regards}}' => $lang['best-regards'] ?? 'Best regards',
        '{{company-team}}' => $lang['company-team'] ?? 'The Datanito Team',
        '{{home}}' => $lang['email-home'] ?? 'Home',
        '{{about}}' => $lang['email-about'] ?? 'About',
        '{{products}}' => $lang['email-ai'] ?? 'Datanito AI',
        '{{contact}}' => $lang['email-contact'] ?? 'Contact',
        '{{privacy}}' => $lang['email-privacy'] ?? 'Privacy',
        '{{company-name}}' => $lang['company-name'] ?? 'Datanito Ltd.',
        '{{company-tagline}}' => $lang['company-tagline'] ?? 'Leading AI-Powered Software and Payment Solutions'
    ];

    foreach ($replacements as $placeholder => $value) {
        $htmlContent = str_replace($placeholder, $value, $htmlContent);
    }

    return $htmlContent;
}


// Function to suggest a related username
function suggestRelatedUsername($username) {
    global $conn;
    // Add a suffix to the username and check if it is available
    $suggestedUsername = $username . mt_rand(1, 9999);
    $sqlCheck = "SELECT * FROM `users` WHERE username = ?";
    $stmtCheck = mysqli_prepare($conn, $sqlCheck);
    mysqli_stmt_bind_param($stmtCheck, "s", $suggestedUsername);
    mysqli_stmt_execute($stmtCheck);
    $resultCheck = mysqli_stmt_get_result($stmtCheck);
    if (mysqli_num_rows($resultCheck) > 0) {
        // Suggested username is already taken, generate a new suggestion recursively
        return suggestRelatedUsername($username);
    }
    return $suggestedUsername;
}
function generateToken() {
    return bin2hex(random_bytes(32)); // Generates a 64-character hexadecimal string
}
function saveTokenInDatabase($user_id, $token, $expires_at, $conn) {

    $sql = "INSERT INTO remember_me_tokens (user_id, token, expires_at) VALUES (?, ?, ?)";
    $stmt = mysqli_prepare($conn, $sql);
    mysqli_stmt_bind_param($stmt, "iss", $user_id, $token, $expires_at);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
}