<?php
include_once includes('config.php');
include_once includes('mailer/send.php');

$user_id = $_SESSION['user']['id'];
$createHost = true;
$country = $_SESSION['country'] ?? 'us';
include_once('includes/language/' . $country . '.php');

// Function to update the database based on payment status
function updateDatabase($invoiceNumber, $status, $conn) {
    // Connection error handling
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }

    // Fetch order numbers
    $getOrderNumbersStmt = $conn->prepare("SELECT order_numbers FROM invoices WHERE invoice_number = ?");
    $getOrderNumbersStmt->bind_param("s", $invoiceNumber);
    $getOrderNumbersStmt->execute();
    $orderNumbersResult = $getOrderNumbersStmt->get_result();
    $row = $orderNumbersResult->fetch_assoc();
    $orderNumbers = json_decode($row['order_numbers'], true);
    $getOrderNumbersStmt->close();

    // Update orders
    foreach ($orderNumbers as $orderNumber) {
        $getOrderStmt = $conn->prepare("SELECT * FROM orders WHERE order_number = ?");
        $getOrderStmt->bind_param("s", $orderNumber);
        $getOrderStmt->execute();
        $orderResult = $getOrderStmt->get_result();

        if ($orderRow = $orderResult->fetch_assoc()) {
            // Determine the new renewal date
            $currentRenewalDate = new DateTime($orderRow['renewal_date']);
            $today = new DateTime();
            $startDate = ($currentRenewalDate > $today) ? $currentRenewalDate : $today;
            $newRenewalDate = $startDate->modify("+{$orderRow['duration']} months")->format('Y-m-d');

            // Update the order
            $updateOrderStmt = $conn->prepare("UPDATE orders SET renewal_date = ?, status = 'Active' WHERE order_number = ?");
            $updateOrderStmt->bind_param("ss", $newRenewalDate, $orderNumber);
            $updateOrderStmt->execute();

            if ($updateOrderStmt->error) {
                error_log("Error updating order {$orderNumber}: " . $updateOrderStmt->error);
                die("Error updating order {$orderNumber}: " . $updateOrderStmt->error);
            }
            $updateOrderStmt->close();
        }
        $getOrderStmt->close();
    }

    // Update the invoice status
    $updateInvoiceStmt = $conn->prepare("UPDATE invoices SET invoice_status = ? WHERE invoice_number = ?");
    $updateInvoiceStmt->bind_param("ss", $status, $invoiceNumber);
    $updateInvoiceStmt->execute();

    if ($updateInvoiceStmt->error) {
        error_log("Error updating invoice {$invoiceNumber}: " . $updateInvoiceStmt->error);
        die("Error updating invoice {$invoiceNumber}: " . $updateInvoiceStmt->error);
    }
    $updateInvoiceStmt->close();
    // Send email with invoice details
    $emailSent = sendInvoiceEmail($invoiceNumber, $conn, $lang);

    // Log result without affecting main process
    if (!$emailSent) {
        error_log("Email sending failed for invoice $invoiceNumber, but process continues.");
    }
}

// Function to create hosting account
function createHostingAccount($user_id, $username, $password, $email, $domain, $package, $order_number, $conn, $whm_host, $whm_user, $whm_pass) {
    // Check if username already exists, if so, add a suffix to make it unique
    $originalUsername = $username;
    $suffix = 1;
    while (usernameExists($username, $conn)) {
        $username = $originalUsername . $suffix;
        $suffix++;
    }

    // Construct the request URL with parameters
    $request_url = "https://$whm_host:2087/json-api/createacct?api.version=1&username=$username&password=$password&domain=$domain&contactemail=$email&plan=$package";

    // Initialize cURL session
    $ch = curl_init();

    // Set cURL options
    curl_setopt($ch, CURLOPT_URL, $request_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // You may need to adjust this option based on your server's SSL configuration
    curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
    curl_setopt($ch, CURLOPT_USERPWD, "$whm_user:$whm_pass");

    // Execute the cURL request
    $response = curl_exec($ch);

    // Check for errors
    if(curl_error($ch)) {
        echo 'Error: ' . curl_error($ch);
    } else {
        // Output the response
        echo $response;

        // Insert account information into the database
        $stmt = $conn->prepare("INSERT INTO hosting_orders (user_id, order_number, username, email, domain, package, password) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("issssss", $user_id, $order_number, $username, $email, $domain, $package, $password);
        $stmt->execute();
        $stmt->close();
    }

    // Close cURL session
    curl_close($ch);
}
function getDomainFromOrder($orderNumber, $conn) {
    // Prepare and execute SQL statement to fetch the domain from the orders table
    $sql = "SELECT name FROM orders WHERE order_number = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $orderNumber);
    $stmt->execute();
    $result = $stmt->get_result();

    // Check if a row is found
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $nameData = json_decode($row['name'], true); // Decode the JSON string into an associative array
        $domain = $nameData['domain']; // Get the domain from the associative array
        $tld = $nameData['tld']; // Get the TLD from the associative array
        return $domain . '.' . $tld; // Return the concatenated domain and TLD
    } else {
        return null; // Return null if no row is found
    }
}


// Function to check if username exists
function usernameExists($username, $conn) {
    $stmt = $conn->prepare("SELECT COUNT(*) AS count FROM hosting_orders WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $count = $row['count'];
    $stmt->close();
    return $count > 0;
}

// Function to create hosting accounts for orders associated with a given invoice number
function createHostingAccountsFromInvoice($invoiceNumber, $conn, $whm_host, $whm_user, $whm_pass) {
    // Step 1: Get order_numbers associated with the invoice_number
    $getOrderNumbersStmt = $conn->prepare("SELECT order_numbers FROM invoices WHERE invoice_number = ?");
    $getOrderNumbersStmt->bind_param("s", $invoiceNumber);
    $getOrderNumbersStmt->execute();
    $getOrderNumbersResult = $getOrderNumbersStmt->get_result();

    // Fetch the result as an associative array
    $row = $getOrderNumbersResult->fetch_assoc();

    // Decode the JSON string to get the array of order_numbers
    $orderNumbers = json_decode($row['order_numbers'], true);

    $getOrderNumbersStmt->close();

    // Step 2: For each order_number, check if it corresponds to web-hosting or business-hosting product and create hosting accounts
    foreach ($orderNumbers as $orderNumber) {
        // Step 3: Get product_id associated with the order_number
        $getProductStmt = $conn->prepare("SELECT product_id FROM orders WHERE order_number = ?");
        $getProductStmt->bind_param("s", $orderNumber);
        $getProductStmt->execute();
        $productResult = $getProductStmt->get_result();

        if ($productRow = $productResult->fetch_assoc()) {
            $productId = $productRow['product_id'];

            // Step 4: Get category associated with the product_id
            $getCategoryStmt = $conn->prepare("SELECT category FROM products WHERE product_id = ?");
            $getCategoryStmt->bind_param("s", $productId);
            $getCategoryStmt->execute();
            $categoryResult = $getCategoryStmt->get_result();

            if ($categoryRow = $categoryResult->fetch_assoc()) {
                $category = $categoryRow['category'];

                // Step 5: If category is web-hosting or business-hosting, create hosting account
                if ($category == 'web-hosting' || $category == 'business-hosting') {
                    // Retrieve user_id associated with the order
                    $userId = getUserIdFromOrder($orderNumber, $conn);

                    // Retrieve user's email to use for account creation
                    $email = getUserEmail($userId, $conn);

                    // Retrieve domain associated with the order
                    $domain = getDomainFromOrder($orderNumber, $conn);

                    // Specify the package name or ID based on the category
                    $package = getPackageFromProduct($productId, $conn);

                    // Generate a username and password (you can use your existing functions for this)
                    $userData = getUserDataByEmail($userId, $conn);

                    // Extract user data
                    $username = generateUsername($userData['username']);
                    $password = generateRandomPassword(); // Generate a random password

                    // Create hosting account
                    createHostingAccount($userId, $username, $password, $email, $domain, $package, $orderNumber, $conn, $whm_host, $whm_user, $whm_pass);
                }
            }
            $getCategoryStmt->close();
        }
        $getProductStmt->close();
    }
}

// Function to retrieve user ID associated with an order
function getUserIdFromOrder($orderNumber, $conn) {
    $stmt = $conn->prepare("SELECT user_id FROM orders WHERE order_number = ?");
    $stmt->bind_param("s", $orderNumber);
    $stmt->execute();
    $result = $stmt->get_result();
    $userId = $result->fetch_assoc()['user_id'];
    $stmt->close();
    return $userId;
}
function getPackageFromProduct($productId, $conn) {
    // Prepare and execute SQL statement to fetch the package from the products table
    $sql = "SELECT package FROM products WHERE product_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $productId);
    $stmt->execute();
    $result = $stmt->get_result();

    // Check if a row is found
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['package']; // Return the package
    } else {
        return null; // Return null if no row is found
    }
}

// Function to retrieve user's email associated with a user ID
function getUserEmail($userId, $conn) {
    $stmt = $conn->prepare("SELECT email FROM users WHERE user_id = ?");
    $stmt->bind_param("s", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $email = $result->fetch_assoc()['email'];
    $stmt->close();
    return $email;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Assuming you have received the payment status and invoice number from the payment process
    $invoiceNumber = $_POST['VerifyEnrollmentRequestId'];
    $status = ($_POST['Status'] == '1') ? 'Paid' : 'Unpaid';
    // Replace this with your actual implementation to retrieve user data from the user table

    // Check if the status is 'Paid' before updating the database
    if ($status == 'Paid') {
        // Update the database
        updateDatabase($invoiceNumber, $status, $conn);

        // Create hosting accounts associated with the invoice
        createHostingAccountsFromInvoice($invoiceNumber, $conn, $whm_host, $whm_user, $whm_pass);
    }
}

// Function to retrieve user data from user table based on email
function getUserDataByEmail($user_id, $conn) {
    $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
    $stmt->bind_param("s", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $userData = $result->fetch_assoc();
    $stmt->close();
    return $userData;
}

// Function to generate a unique username
function generateUsername($baseUsername) {
    // You can implement your logic to make the username unique
    // For example, you can append numbers or random strings to the base username
    return $baseUsername . rand(100, 999); // Appending a random 3-digit number
}

// Function to generate a random password
function generateRandomPassword($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*()-_=+';
    $password = '';
    $maxIndex = strlen($characters) - 1;
    for ($i = 0; $i < $length; $i++) {
        $password .= $characters[rand(0, $maxIndex)];
    }
    return $password;
}

// Function to send email with invoice and order details
function sendInvoiceEmail($invoiceNumber, $conn, $lang) {
    // Fetch invoice and order details
    $getInvoiceStmt = $conn->prepare("SELECT * FROM invoices WHERE invoice_number = ?");
    $getInvoiceStmt->bind_param("s", $invoiceNumber);
    $getInvoiceStmt->execute();
    $invoiceResult = $getInvoiceStmt->get_result();
    $invoiceData = $invoiceResult->fetch_assoc();
    $getInvoiceStmt->close();


    if (!$invoiceData) {
        error_log("Invoice $invoiceNumber not found");
        return false;
    }

    // Decode order numbers
    $orderNumbers = json_decode($invoiceData['order_numbers'], true);
    $orderDetails = "";
    $isPaid = ($invoiceData['invoice_status'] == "Paid");

    foreach ($orderNumbers as $orderNumber) {
        $getOrderStmt = $conn->prepare("SELECT * FROM orders WHERE order_number = ?");
        $getOrderStmt->bind_param("s", $orderNumber);
        $getOrderStmt->execute();
        $orderResult = $getOrderStmt->get_result();

        if ($orderRow = $orderResult->fetch_assoc()) {
            $productDetails = getProductDetails($orderRow['product_id'], $conn);

            $orderDetails .= "<p>" . $lang['order-number'] . ": {$orderRow['order_number']}</p>";
            $orderDetails .= "<p>" . $lang['product-name'] . ": {$productDetails['name']}</p>";
            $orderDetails .= "<p>" . $lang['order-status'] . ": {$orderRow['status']}</p>";
            $orderDetails .= "<p>" . $lang['renewal-date'] . ": {$orderRow['renewal_date']}</p><hr>";

            if ($orderRow['status'] == 'paid') {
                $isPaid = true; // Mark order as paid
            }
        }

        $getOrderStmt->close();
    }

    $user_id = $invoiceData['user_id'];
    $userData = getUserDataByEmail($user_id, $conn);

    // Email preparation
    $emailCtaLink = url('/account/billing/invoices/' . $invoiceNumber);

    if ($isPaid) {
        $emailSubjectUser = $lang['email-paid-subject'];
        $emailContentUser = getEmailContent($emailCtaLink, $orderDetails, $userData, $lang, $lang['email-paid-headline']);
    } else {
        $emailSubjectUser = $lang['email-unpaid-subject'];
        $emailContentUser = getEmailContent($emailCtaLink, $orderDetails, $userData, $lang, $lang['email-unpaid-headline']);
    }

    // Send email
    try {
        $successAdmin = sendEmail("info@datanito.com", $lang['email-admin-subject'], $emailContentUser, '');
        $successUser = sendEmail($userData['email'], $emailSubjectUser, $emailContentUser, '');

        return $successAdmin && $successUser;
    } catch (Exception $e) {
        error_log("Email sending error: " . $e->getMessage());
        error_log("Failed to send email for Invoice $invoiceNumber to {$invoiceData['user_id']}");
        return false;
    }
}

function getEmailContent($emailCtaLink, $orderDetails, $userData, $lang, $headline) {
    $htmlContent = file_get_contents('includes/emails/system_email.php');

    $htmlContent = str_replace('{{cta-link}}', $emailCtaLink, $htmlContent);
    $htmlContent = str_replace('{{user-name}}', $userData['name'], $htmlContent);
    $htmlContent = str_replace('{{hello}}', $lang['email-hello'], $htmlContent);
    $htmlContent = str_replace('{{headline}}', $headline, $htmlContent);
    $htmlContent = str_replace('{{email-p1}}', $orderDetails, $htmlContent);
    $htmlContent = str_replace('{{cta}}', $lang['email-cta'], $htmlContent);
    $htmlContent = str_replace('{{email-p2}}', $lang['email-footer'], $htmlContent);
    $htmlContent = str_replace('{{email-p3}}', '', $htmlContent);
    $htmlContent = str_replace('{{email-copyright}}', $lang['email-copyright'], $htmlContent);

    return $htmlContent;
}

function getProductDetails($productId, $conn) {
    // Determine the language and currency settings
    $country = $_SESSION['country'] ?? 'us';
    $currency = isset($_SESSION['currency']) ? strtolower($_SESSION['currency']) : 'usd'; // Ensure currency code is in lowercase

    // Define the columns based on the language and currency settings
    $nameColumn = ($country === 'tr') ? 'name_tr' : 'name';
    $descriptionColumn = ($country === 'tr') ? 'description_tr' : 'description';
    $priceColumn = 'price_' . $currency; // Ensure currency code is in lowercase

    // Query to fetch product data
    $query = "SELECT product_id, $nameColumn AS name, $descriptionColumn AS description, picture, $priceColumn AS price FROM products WHERE product_id = ?";

    // Use prepared statements to prevent SQL injection
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $productId);
    $stmt->execute();

    $result = $stmt->get_result();
    $productData = $result->fetch_assoc();

    $stmt->close();

    return $productData;
}
